; ===========================================================================
; ---------------------------------------------------------------------------
; Test ROM
; ---------------------------------------------------------------------------

		include "Z80 Language.asm"
		include	"Equates.asm"
		include	"Macros.asm"

; ===========================================================================
; ---------------------------------------------------------------------------
; Start of ROM data
; ---------------------------------------------------------------------------

StartRom:	dc.l	R_StackSP				; stack/VDP reg address
		moveq	#$00,d4					; clear d4
		ori.b	#.SetupList,d4				; a filler instruction, must be ori.b, must be d4 (The immediate value can be whatever necessary though).

	; --- TMSS ---

		movea.w	d4,a0					; set address of setup list
		movem.l	(a0)+,d2/d7/a2/a4-a6			; VDP reg counters, DMA write, control ports address, & VDP data and control ports
		movem.w	(a0)+,d0/d1/d5-d6/a3			; load number of entries, 80XX value, 0100 for advancing (and for Z80 later), VDP reg RAM
		and.b	-$07(a4),d0				; get only version number
		beq.s	.NoTMSS					; if it's version 0, branch
		move.l	(ConsoleName).w,($3FFF-$07)(a4)		; save SEGA to TMSS chip

	.NoTMSS:

	; --- VDP DMA wait/Latch reset ---

		move.w	(a6),ccr				; load status (this resets the 2nd write flag too)
		bvs.s	*-2					; if the VDP DMA busy flag was set (bit 1), branch to wait til finished...

	; --- VDP Register Setup & VRAM Clear ---

		lea	(sp),a1					; load beginning of VDP register RAM

	.NextVDP:
		addq.b	#(.DMAE-.DMAS)+1,d1			; increase count correctly for normal VDP registers/set size for DMA registers)

	.SetupVDP:
		move.w	d2,(a6)					; set VDP register
		move.w	d2,(a1)+				; ...and save to RAM storage
		add.w	d5,d2					; advance register ID
		move.b	(a0)+,d2				; load next register value
		subq.b	#$01,d1					; decrease register count (LOWER BYTE ONLY, Upper byte is needed for PSG mute counter)
		bpl.s	.SetupVDP				; if still counting, repeat
		move.w	a5,(a1)+				; clear "movep" pad slot / "word" bug slot
		add.w	d6,d2					; advance to DMA register slot/ID
		neg.w	d5					; reverse register increment direction
		bmi.s	.NextVDP				; if not positive, we haven't done DMA registers yet, so loop for them
		move.l	d7,(a6)					; set DMA fill destination
		move.w	a5,(a5)					; set DMA fill value (0000)

	; --- 68k RAM Clear ---

		move.l	(a3)+,a1 ; a1 = $8E008F01 (ROM 8F01)	; set to clear ROM space (just to waste time)
		tst.w	(a4)+					; has reset occurred? (i.e. port A been initialised already)
		bne.s	.ClearRAM				; if so, (fail to) clear ROM to waste CPU time for DMA fill
		lea	(sp),a1  ; a1 = $FFFFFFXX (RAM)		; set to clear RAM space instead

	.ClearRAM:
		sf.b	-(a1)					; clear RAM (fail to clear ROM to waste time)
		move.w	a1,d4					; have we reached 0000 yet?
		bne.s	.ClearRAM				; if not, continue until clear

	; --- Loading Z80 related data (for Part 2) ---

		; On power on, Z80 reset is apparently on, but on
		; soft reset it may be off, so might as well put
		; it on anyway to ensure YM2612 is reset if anything.

		lea	$11F6(a4),a1				; load Z80 BUS address area
		move.w	d5,-$100(a1)				; request Z80 stop (ON)
		move.w	a5,(a1)					; request Z80 reset (ON) (resets YM2612)

	; --- CRAM/VSRAM Clear ---

		lsr.b	#$03,d7					; move DMA bit to VSRAM address bit
		addq.w	#$01,-(a3)				; reset increment mode for 68k RAM
		move.w	(a3),(a6)				; '' (and now for VDP)

	.ClearCRAM:
		move.l	d7,(a6)					; set VDP to VSRAM write mode
		sf.b	d7					; set to clear $100 words (won't matter due to wrapping, but need to waste time to let Z80 finish reset)

	.ClearVSRAM:
		move.w	a5,(a5)					; clear VSRAM/CRAM (done by word to slow the 68k down enough to stop the Z80 in time)
		subq.b	#$01,d7					; decrease counter
		bne.s	.ClearVSRAM				; if not finished, loop and keep clearing
		move.w	d5,(a1)					; request Z80 reset (OFF) (CANNOT WRITE TO Z80 RAM WHILE RESET IS ON!  (I tried ;-;))
		move.b	(a0)+,(a2)+				; copy di/jp instruction to Z80 RAM
		neg.l	d7					; change address from VSRAM to CRAM
		bmi.s	.ClearCRAM				; if we're now writing to CRAM (negative $C0000000), then repeat for CRAM

; ---------------------------------------------------------------------------
; Setup information
; ---------------------------------------------------------------------------
;	 	align	$0070
; ---------------------------------------------------------------------------

		dc.w	%0100101011110000			; "tas.b $??(a0,??.?)" (should be "tas.b $06(a0,sp.l)")
		dc.w	(R_HBlankJmp&$FFFF)			; H-blank address
		swap	d7					; get $4000 in lower word for address checking with Z80
		bra.s	.Continue				; jump to part 2 code
		dc.l	R_VBlankJmp				; V-blank address

	.SetupList:
		dc.l	$00408000|%00000100		; d2	; control port init & 00LH 01CD - Leftover SMS bar (0N|1Y) | H-Interrupt (0N|1Y) | H,V Counter (0N|1Y) | Disable video signal (0N|1Y)
		dc.l	$40000080			; d7	; DMA Fill destination
		dc.l	$00A02000			; a2	; Z80 RAM address (mirror access 2000 to 4000)
		dc.l	$00A10008			; a4	; control ports (start on port A)
		dc.l	$00C00000			; a5	; VDP data port
		dc.l	$00C00004			; a6	; VDP control port

		dc.w	$000F				; d0
		dc.w	$FE00|(.VDPE-.VDPS)-(.DMAE-.DMAS) ; d1	; standard VDP register count (and upper byte for PSG mute counter)
		dc.w	$0100				; d5	; VDP Reg increment value & opposite initialisation flag for Z80
		dc.w	$0400				; d6	; for advancing VDP registers to start at last DMA reg
		dc.w	R_VDPReg8E&$FFFF		; a3	; RAM address where VDP auto-increment value is stored

	.VDPS:	dc.b	%00010100				; SDVM P100 - SMS mode (0N|1Y) | Display (0N|1Y) | V-Interrupt (0N|1Y) | DMA (0N|1Y) | V-resolution (0-1C|1-1E)
		dc.b	((($C000)>>$0A)&$FF)			; 00FE DCBA - Scroll Plane A Map Table VRam address
		dc.b	((($D000)>>$0A)&$FF)			; 00FE DCB0 / 00FE DC00 (20 Resolution) - Window Plane A Map Table VRam address
		dc.b	((($E000)>>$0D)&$FF)			; 0000 0FED - Scroll Plane B Map Table VRam address
		dc.b	((($B800)>>$09)&$FF)			; 0FED CBA9 / 0FED CBA0 (20 Resolution) - Sprite Plane Map Table VRam address
		dc.b	$00					; Unknown/Unused Register
		dc.b	$00					; 00PP CCCC - Backdrop Colour: Palette Line 0/Colour ID 0
		dc.b	$00					; Unknown/Unused Register
		dc.b	$00					; Unknown/Unused Register
		dc.b	$DF					; 7654 3210 - H-Interrupt Register
		dc.b	%00000100				; 0000 EVHH - External Interrupt (0N|1Y) | V-Scroll: (0-Full|1-2Celled) | H-Scroll: (00-Full|10-Celled|11-Sliced)
		dc.b	%10000001				; APHE SIIB - (AB) H-resol (0N|1Y) | Pixel int (0N|1Y) | H-sync (0N|1Y) | Extern-pix (0N|1Y) | S/H (0N|1Y) | Interlace (00N|01Y|11-Split) | (AB) H-resol (0-20|1-28)
		dc.b	((($BC00)>>$0A)&$FF)			; 00FE DCBA - Horizontal Scroll Table VRam address
		dc.b	$00					; Unknown/Unused Register
		dc.b	$01					; 7654 3210 - Auto Increment: By 1 (for DMA fill)
		dc.b	%00000001				; 00VV 00HH - Plane Y Size (00-20|01-40|11-80) | Plane X size (00-20|01-40|11-80)
		dc.b	$00					; 7654 3210 - Window Horizontal Position
	.VDPE:	dc.b	$00					; 7654 3210 - Window Vertical Position

	.DMAS:	dc.b	$80					; D654 3210 - DMA Fill Source (800000 Fill mode)
		dc.b	$00					; FEDC BA98 - ''
		dc.b	$00					; 7654 3210 - ''
		dc.b	$FF					; FEDC BA98 - DMA Fill Size (FFFF bytes)
	.DMAE:	dc.b	$FF					; 7654 3210 - ''

		dc.b	$40					; control port init 2

		dc.b	$F3,$C3					; di and jp instructions for Z80

; ---------------------------------------------------------------------------
; Continue (Part 2)
; ---------------------------------------------------------------------------

	.RunReset:
		jmp	(ResetGame).l				; jump to reset routine

	.Continue:

	; --- PSG Mute ---

		; d1 contains FEFF, the below will write all X7 and XF bytes
		; to the PSG, ensuring the last byte the PSG receives of each
		; channel is a volume of F (9F, BF, DF, and FF).
		; Hopefully it'll be quick enough that no tone is played...

	.MutePSG:
		move.b	d1,$0D(a6)				; mute PSG channel
		addq.w	#$08,d1					; advance up gradually to each PSG volume channel and force volume F
		bcc.s	.MutePSG				; if we haven't passed FF yet, loop

	; --- Z80 Setup ---


	.ClearZ80:
		move.b	d7,(a2)+				; clear Z80 space
		cmp.w	a2,d7					; have we reached end of Z80 RAM mirror?
		bne.s	.ClearZ80				; if we haven't, keep going...

	; --- ROM Checksum ---


		movea.w	a5,a2					; starting ROM address to checksum from
		; d7 contains $00004000, so the checksum value
		; must be +$00004000 too, no other registers are clear...

	.CheckSum:
		add.l	(a2)+,d7				; add a long-word of checksum
		sne.b	d0					; set d0 0000 or 00FF depending on sum being 0
		cmpa.l	(ROMFinish).w,a2			; have we passed the end of ROM address yet?
		bmi.s	.CheckSum				; if not, keep adding
		lsr.b	#$04,d0					; get only red
		move.w	d0,(a5)					; write backdrop colour
		bne.s	*					; if the colour wasn't black, loop endlessly

		move.w	a5,-$100(a1)				; request Z80 stop (OFF)

	; --- I/O port initialisation and jump to game code ---

		move.l	d2,(a4)					; initialise ports B and C
		tst.w	-(a4)					; has port A been initialised?  (i.e. are we resetting?)
		bne.s	.RunReset				; if so, run reset routine instead
		move.w	(a0),(R_MainJmp).w			; setup main routine "jmp" instruction
		move.w	(a0),(R_HBlankJmp).w			; setup H-blank "jmp" instruction
		move.w	(a0),(R_VBlankJmp).w			; setup V-blank "jmp" instruction
		move.w	d2,(a4)					; initialise port A (last one)
		jmp	(StartGame).l				; run normal power on routine

		dc.l	$00000000				; sum round up (account for +$00004000 in d7)

; ---------------------------------------------------------------------------
; Mega Drive specific data
; ---------------------------------------------------------------------------
;		align	$0100
; ---------------------------------------------------------------------------

ConsoleName:	dc.b	"SEGA MEGA DRIVE "
ProductDate:	dc.b	"(C)???? YYYY.MMM"
TitleLocal:	dc.b	"Engine Test                                     "
TitleInter:	dc.b	"Engine Test                                     "
SerialNumber:	dc.b	"GM XXXXXXXXXXX"
Checksum:	dc.w	$0000
IOSupport:	dc.b	"J               "
ROMStart:	dc.l	StartRom
ROMFinish:	dc.l	FinishRom-1
RAMStart:	dc.l	$00FF0000
RAMFinish:	dc.l	$00FFFFFF
SupportRAM:	dc.b	'    '		;'RA',%11111000,%00100000	; 1E1TT000 DDD0000 ; E = Erase (0 No | 1 Yes) TT = Type (00 both | 01 serial EEPROMS (RAMs with 4-bit data bus, etc...) | 10 even | 11 odd) DDD = Device type (001 SRAM | 010 EEPROM | rest unused)
SRAMStart:	dc.l	'    '		; $00200001
SRAMFinish:	dc.l	'    '		; $002001FF			; $00200001 (<- for modem)
SupportModem:	dc.b	"            "	; "MO*name*XX.YZZ   "
Notes:		dc.b	"                                        "
ProductRegion:	dc.b	"JUE             "

; ===========================================================================
; ---------------------------------------------------------------------------
; Start of game
; ---------------------------------------------------------------------------

ResetGame:
		move.l	#$C0000000,(a6)
		move.w	#$0E00,(a5)
		bra.w	*

StartGame:
		move.l	#$C0000000,(a6)
		move.w	#$00E0,(a5)

	; --- Z80 ---

		lea	Z80(pc),a0					; load Z80 ROM data
		lea	($A00000).l,a1					; load Z80 RAM space address
		move.w	#(Z80_End-Z80)-$01,d1				; size of DualPCM
		move.w	#$0100,($A11100).l				; request Z80 stop (ON)
		move.w	#$0100,($A11200).l				; request Z80 reset (OFF)
		btst.b	#$00,($A11100).l				; has the Z80 stopped yet?
		bne.s	*-$08						; if not, branch

	.LoadZ80:
		move.b	(a0)+,(a1)+					; copy Dual PCM to Z80 RAM
		dbf	d1,.LoadZ80					; repeat til done

		move.w	#$0000,($A11200).l				; request Z80 reset (ON)
		moveq	#$7F,d1						; set delay time
		dbf	d1,*						; no way of checking for reset, so a manual delay is necessary
		move.w	#$0000,($A11100).l				; request Z80 stop (OFF)
		move.w	#$0100,($A11200).l				; request Z80 reset (OFF)

	; --- GMS request ---

		lea	(GSM_Sample).l,a0				; load GSM sample
		jsr	SetSampleGSM					; set to play GSM sample

		move.l	#VBlank,(R_VBlankRout).w			; set V-blank routine to run
		ori.b	#%01100000,(R_VDPReg81+1).w			; SDVM P100 - SMS mode (0N|1Y) | Display (0N|1Y) | V-Interrupt (0N|1Y) | DMA (0N|1Y) | V-resolution (0-1C|1-1E)
		move.w	(R_VDPReg81).w,($C00004).l			; enable V-blank

ScreenLoop:
		bsr.w	WaitVBlank					; wait for V-blank
		bra.w	ScreenLoop					; loop

; ===========================================================================
; ---------------------------------------------------------------------------
; Waiting for V-blank
; ---------------------------------------------------------------------------

WaitVBlank:
		st.b	(R_VBlankRan).w					; set 68k as ready for V-blank
		move.w	#$2300,sr					; enable interrupts
	.Wait:	tst.b	(R_VBlankRan).w					; has V-blank marked itself as ran yet?
		bne.s	.Wait						; if not, loop and wait
		rts							; done

; ===========================================================================
; ---------------------------------------------------------------------------
; V-blank
; ---------------------------------------------------------------------------

VBlank:
		tst.b	(R_VBlankRan).w					; was the 68k late?
		beq.s	.Late						; if so, skip
		sf.b	(R_VBlankRan).w					; mark as ran

		jsr	DecodeGSM					; decode a block of GSM

	.Late:
		rte							; done

; ===========================================================================
; ---------------------------------------------------------------------------
; GSM 6.10 68k driver - MarkeyJester
; ---------------------------------------------------------------------------
		rsset $FFFF8000
GSM:		rs.b 	0
GSM_Current:	rs.l	1						; pointer to current sample
GSM_LARppCur:	rs.w	1						; pointer to current LAR'' table
GSM_LARppPre:	rs.w	1						; pointer to previous LAR'' table
GSM_LARppj:	rs.w	8						; LAR''(j) 1 to 8
GSM_LARppjm:	rs.w	8						; LAR''(j-1) 1 to 8
GSM_df:		rs.w	40*4						; dp [-160 to 0]
GSM_dp:		rs.w	40*4						; dp [0 to 160]
GSM_v:		rs.w	9						; v [0 to 9]
GSM_msr:	rs.w	1						; msr = 0 initial value
GSM_Reset:	rs.b	0	; everything above is reset to 0 on new sample, below no need
GSM_rrp:	rs.w	8						; rrp [1 to 8]
GSM_LoadxMc:	rs.l	1						; next routine for loading xMc bits
GSM_Output:	rs.b	160						; 160 samples

; ===========================================================================
; ---------------------------------------------------------------------------
; Requesting a GSM sample to play
; ---------------------------------------------------------------------------

SetSampleGSM:
		lea	(GSM).w,a1					; load GSM RAM
		moveq	#$00,d0						; clear d0
		move.l	d0,(a1)+					; stop current GSM from playing
		move.w	#GSM_LARppj&$FFFF,(a1)+				; set starting LAR''(j) buffer
		move.w	#GSM_LARppjm&$FFFF,(a1)+			; set starting LAR''(j-1) buffer
		moveq	#((GSM_Reset-GSM_LARppj)/8)-1,d1		; reset all GSM buffers
	.Reset:	move.l	d0,(a1)+					; reset GSM buffers
		move.l	d0,(a1)+					; ''
		dbf	d1,.Reset					; '' repeat until done
	if ((GSM_Reset-GSM_LARppj)&4)<>0
		move.l	d0,(a1)+					; reset last longword
	endif
	if ((GSM_Reset-GSM_LARppj)&2)<>0
		move.w	d0,(a1)+					; reset last word
	endif
	if ((GSM_Reset-GSM_LARppj)&1)<>0
		move.b	d0,(a1)+					; reset last byte
	endif
		move.l	a0,(GSM_Current).w				; set sample as ready to play
		rts							; ''

; ===========================================================================
; ---------------------------------------------------------------------------
; Playing back (decoding a block of) GSM - To be called every V-blank
; ---------------------------------------------------------------------------

DB_FinishSample:
		clr.l	(GSM_Current).l					; mark no sample to play

DB_NoSample:
		rts							; done

DecodeGSM:
		move.l	(GSM_Current).w,d0				; load sample
		beq.s	DB_NoSample					; if not sample has loaded, cancel
		movea.l	d0,a0						; set address of sample

; ---------------------------------------------------------------------------
; Loading and decoding of the quantised Log.-Area Ratio coefficients (LARc)
; LAR''(i) = ( LARc(i) - B(i) )/ A(i)
; --- expected registers ----------------------------------------------------
; hl = Start of GSM block:
;
;	DDDD1111
;	11222222
;	33333444
;	44555566
;	66777888
; ---------------------------------------------------------------------------

		lea	(LARpp1).l,a1					; load LARc bit to LARpp(j) tables
		movea.w	(GSM_LARppCur).w,a2				; laod current LARpp(j) table

		move.w	#$0200,d0					; reset LARc table address
		moveq	#$00,d1						; reset LARpp(j) table address
		move.w	#$0100,d2					; prepare table

		move.b	(a0)+,d0					; load DDDD1111
		bpl.s	DB_FinishSample					; if positive, then likely no magic D, so end of sample
		move.b	(a1,d0.w),d1					; load 01111000
		add.w	d2,d0						; next table
		move.b	(a0)+,d0					; load 11222222
		add.b	(a1,d0.w),d1					; make 01111110
		add.w	d2,d0						; next table
		move.w	(a1,d1.w),(a2)+					; save LAR''(j)[1]

		move.b	(a1,d0.w),d1					; load 02222220
		add.w	d2,d0						; next table
		move.w	(a1,d1.w),(a2)+					; save LAR''(j)[2]

		move.b	(a0)+,d0					; load 33333444
		move.b	(a1,d0.w),d1					; load 00333330
		add.w	d2,d0						; next table
		move.w	(a1,d1.w),(a2)+					; save LAR''(j)[3]

		move.b	(a1,d0.w),d1					; load 00444000
		add.w	d2,d0						; next table
		move.b	(a0)+,d0					; load 44555566
		add.b	(a1,d0.w),d1					; make 00444440
		add.w	d2,d0						; next table
		move.w	(a1,d1.w),(a2)+					; save LAR''(j)[4]

		add.w	d2,d1						; next LAR''(j) conversion table

		move.b	(a1,d0.w),d1					; load 00055550
		add.w	d2,d0						; next table
		move.w	(a1,d1.w),(a2)+					; save LAR''(j)[5]

		move.b	(a1,d0.w),d1					; load 00066000
		add.w	d2,d0						; next table
		move.b	(a0)+,d0					; load 66777888
		add.b	(a1,d0.w),d1					; make 00066660
		add.w	d2,d0						; next table
		move.w	(a1,d1.w),(a2)+					; save LAR''(j)[6]

		move.b	(a1,d0.w),d1					; load 00007770
		add.w	d2,d0						; next table
		move.w	(a1,d1.w),(a2)+					; save LAR''(j)[7]

		move.b	(a1,d0.w),d1					; load 00008880
		move.w	(a1,d1.w),(a2)+					; save LAR''(j)[8]

; ---------------------------------------------------------------------------
; frames 1 to 4
; ---------------------------------------------------------------------------

		lea	($A00000+Buffer+1).l,a5				; Z80 address of buffer flag
		move.w	#$0100,($A11100).l				; request Z80 stop (ON)
		btst.b	#$00,($A11100).l				; has the Z80 stopped yet?
		bne.s	*-$08						; if not, branch
		move.b	(a5)+,d0					; load buffer flag
		move.w	#$0000,($A11100).l				; request Z80 stop (OFF)
		lea	($A00000+BufferA).l,a5				; load buffer A
		tst.b	d0						; is buffer A being played?
		bmi.s	.UseA						; if not, write to A
		lea	($A00000+BufferB).l,a5				; load buffer B

	.UseA:

	;	lea	(GSM_Output).l,a5				; load output buffer
		lea	(GSM_dp).w,a2					; start of dp slot
		lea	(GSM_df).w,a4					; load to a4 for storing x brp result with xMc
		moveq	#$04-1,d7					; 4 frames
		move.l	#$00004000,d4					; prepare roundup value

FrameLoop:
		lea	(a2),a1						; load dp frame slot
		lea	40*2(a2),a2					; advance to next 40 slot (for next frame)
		move.l	a2,-(sp)					; store dp slot
		move.w	d7,-(sp)					; store frame count so d7 can be used for rounding
		move.l	d4,d7						; ''

	; ---------------------------------------------------------------------------
	; Unpacking Nc, bc, Mc, xmaxc
	; ---------------------------------------------------------------------------
	; LTP lag (Nc) is never encoded below 40 or above 120.  The decoder has measures
	; to reuse the previous Nc if the new one is outside that range.  But I suspect
	; that's for data loss handling (i.e. mobile), we won't have that issue here
	; so handling out of range won't be necessary.
	; ---------------------------------------------------------------------------
	; The xmaxc and xMc together make up the APCM inverse quantization, the result
	; of these are added to the dp as part of reconstructing the short term residual
	; signal.  The xmaxc is used to calculate the exponent and mantissa, of which
	; mantissa is used to collect the FAC (Normalised direct mantissa), multiplied
	; with xMc gets the ep to later be added to dp.
	;
	; Since there are only 8 possible xMc's and $40 possible xmaxc's, we can condense
	; these down to a LUT quite nicely, and eliminate a huge amount of calculations
	; including a mult_r.
	; ---------------------------------------------------------------------------
	; hl = Start of GSM frame:
	;
	;	NNNNNNNB
	;	BMMXXXXX
	;	X1112223
	; ---------------------------------------------------------------------------

		moveq	#$00,d0						; load LTP lag (Nc) as x 2
		move.b	(a0)+,d0					; ''
		moveq	#$01,d2						; get first bit of LTP gain (bc)
		and.b	d0,d2						; ''
		eor.b	d2,d0						; get only Nc x 2
	; d0.w = Nc x 2
		suba.w	d0,a1						; move dp back
	; a1.l = dp [(frame * 40) - Nr]

		moveq	#$00,d4						; load BMMXXXXX
		move.b	(a0)+,d4					; '' (second bit bc, Mc, and xmaxc)
		moveq	#$FFFFFF80,d3					; load last bit of xmaxc
		and.b	(a0),d3						; ''
		add.b	d3,d3						; shift xmaxc bit over
		addx.b	d4,d4						; '' ...shift second bit of bc over
		addx.b	d2,d2						; ''
		lea	QLB(pc),a2					; load QLB table
		add.b	d2,d2						; load correct brp value
		move.w	(a2,d2.w),d2					; ''
	; d2.w = brp

		add.b	d4,d4						; send first bit of Mc to other register
		addx.b	d3,d3						; ''
		add.b	d4,d4						; send second bit of MC to other register
		addx.b	d3,d3						; ''
		ext.w	d3						; clear upper byte
	; d3.w = Mc

		add.w	d4,d4						; arrange xmaxc bits for table 000000XX XXXX0000
		add.w	d4,d4						; ''
	; d4.w = xmaxc << 4

	; ---------------------------------------------------------------------------
	; Single sub-frame loop (40 samples per frame)
	; ---------------------------------------------------------------------------
	; This section deals with unpacking 40 samples out of the xMc data in this
	; frame, it handles the xMc unpacking and converting with xmaxc to create the
	; ep value, and sum this with the brp (bc) we collected earlier, multiplied
	; with the previous dp section 
	; ---------------------------------------------------------------------------

		lea	epval(pc),a3					; load ep values table
		move.w	#40*2*4,d5					; prepare index register
		moveq	#40-1,d6					; 40 signal samples per frame

		move.l	#LoadxMc1,(GSM_LoadxMc).w			; store first xMc bit loading routine

dpSampleLoop:

		; ---------------------------------------------------------------------------
		; We're unpacking 40 samples in a loop, however, only 13 of those times are
		; xMc's unpacked and summed, and they're done in increments of 3 (Mc + i*3)
		; so instead of two loops, we'll simply skip xMc unpacking 2 out of 3, and
		; we'll use Mc to dictate the starting point (see above loading Mc into SkipxMc+1)
		;
		; please note, by "sample" I don't mean final 100% result, I mean simply the
		; "dp" sample result (i.e. reconstructed short term residual signal)
		;
		; The next loop after at k_start to k_end will be the real sample loop
		; ---------------------------------------------------------------------------

		moveq	#$00,d1						; no ep value to add
		dbf	d3,.SkipxMc					; skip xMc coefficient writing until it's time
		moveq	#$03-1,d3					; reset for next 3rd i
		movea.l	(GSM_LoadxMc).w,a2				; load xMc routine
		jsr	(a2)						; run xMc routine
		or.w	d4,d1						; save xmaxc bits with xMc bits
		move.w	(a3,d1.w),d1					; load correct ep value

	.SkipxMc:
		move.w	(a1)+,d0					; load dp [ - Nr + k]
		muls.w	d2,d0						; multiply by brp
		add.l	d7,d0						; round up
		add.l	d0,d0						; get >> 1
		swap	d0						; get >> $F
		add.w	d1,d0						; add ep value
		move.w	d0,(a4,d5.w)					; save to current dp 0 to 40 (-160 to -120)
		move.w	d0,(a4)+					; save to underflow dp 0 to 40
		dbf	d6,dpSampleLoop					; repeat for all 40 signals
		move.l	d7,d4						; load rounding value back to d4
		move.w	(sp)+,d7					; restore frame count

	; ---------------------------------------------------------------------------
	; convert the LARpp to rrp here
	; ---------------------------------------------------------------------------

		movea.w	(GSM_LARppCur).w,a1				; load current LAR''(j)
		lea	(GSM_rrp).w,a3					; load reflective LAR storage
		moveq	#8-1,d6						; do 8 LAR's

		move.w	d7,d0
		add.w	d0,d0
		jmp	GSM_LAR(pc,d0.w)
GSM_LAR:	bra.s	GSM_LAR4					; d7 = 0
		bra.s	GSM_LAR3					; d7 = 1
		bra.s	GSM_LAR2					; d7 = 2
									; d7 = 3

		; -------------------------------------------------------
		; Interpolation of LARpp (1st coefficient set (0 to 12))
		;	for (int i = 1; i <= 8; i++)
		;	{
		;		short LAR = (LARpp [pre] [i] >> 2) + (LARpp [cur] [i] >> 2) + (LARpp [pre] [i] >> 1);
		;		rrp [i] = GSM_ReflectLAR (LAR);
		;	}
		;	k_start = 0;
		;	k_end = 12;
		; -------------------------------------------------------

GSM_LAR1:
		movea.w	(GSM_LARppPre).w,a2				; load previous LAR''(j-1)

		; --- LAR''(j-1) >> 1 ---
		; All 4 frames of LAR reflection use the following:
		;   LAR''(j) >> 2     LAR''(j) >> 1     LAR''(j)
		;   LAR''(j-1) >> 2   LAR''(j-1) >> 1
		; Since LAR''(j-1) (previous) is never used in any of the
		; 4 frames, we'll shift it right once here, and store it back
		; again, to save doing shift right >> 2 for the other frames
		; -----------------------

	.Nexti:
		move.w	(a2),d0						; load previous LAR
		asr.w	#$01,d0						; get >> 1
		move.w	d0,(a2)+					; store (other frames don't need non-shift previous either)
		move.w	d0,d1						; get >> 2
		asr.w	#$01,d1						; ''
		add.w	d1,d0						; (pre >> 2) + (pre >> 1)
		move.w	(a1)+,d1					; load current LAR
		asr.w	#$02,d1						; get >> 2
		add.w	d1,d0						; (cur >> 2) + (pre >> 2) + (pre >> 1)
		bsr.w	GSM_ReflectLAR					; perform reflection of LAR
		move.w	d0,(a3)+					; save to rrp
		dbf	d6,.Nexti					; repeat for all 8

		lea	(GSM_dp+0*2).w,a1				; k_start
		moveq	#(12-0),d6					; k_end
		bra.w	GSM_DoneLAR

		; -------------------------------------------------------
		; Interpolation of LARpp (2nd coefficient set (13 to 26))
		;	for (int i = 1; i <= 8; i++)
		;	{
		;		short LAR = (LARpp [pre] [i] >> 1) + (LARpp [cur] [i] >> 1);
		;		rrp [i] = GSM_ReflectLAR (LAR);
		;	}
		;	k_start = 13;
		;	k_end = 26;
		; -------------------------------------------------------

GSM_LAR2:
		movea.w	(GSM_LARppPre).w,a2				; load previous LAR''(j-1)

		; --- LAR''(j-1) >> 1 ---
		; Already shifted right once on first frame
		; -----------------------

	.Nexti:
		move.w	(a1)+,d0					; load current LAR
		asr.w	#$01,d0						; get >> 1
		add.w	(a2)+,d0					; (cur >> 1) + (pre >> 1)
		bsr.w	GSM_ReflectLAR					; perform reflection of LAR
		move.w	d0,(a3)+					; save to rrp
		dbf	d6,.Nexti					; repeat for all 8

		lea	(GSM_dp+13*2).w,a1				; k_start
		moveq	#(26-13),d6					; k_end
		bra.w	GSM_DoneLAR

		; -------------------------------------------------------
		; Interpolation of LARpp (3rd coefficient set (27 to 39))
		;	for (int i = 1; i <= 8; i++)
		;	{
		;		short LAR = (LARpp [pre] [i] >> 2) + (LARpp [cur] [i] >> 2) + (LARpp [cur] [i] >> 1);
		;		rrp [i] = GSM_ReflectLAR (LAR);
		;	}
		;	k_start = 27;
		;	k_end = 39;
		; -------------------------------------------------------

GSM_LAR3:
		movea.w	(GSM_LARppPre).w,a2				; load previous LAR''(j-1)

		; --- LAR''(j-1) >> 1 ---
		; Already shifted right once on first frame
		; -----------------------

	.Nexti:
		move.w	(a1)+,d0					; load current LAR
		asr.w	#$01,d0						; get >> 1
		move.w	d0,d1						; get >> 2
		asr.w	#$01,d1						; ''
		add.w	d1,d0						; (cur >> 2) + (cur >> 1)
		move.w	(a2)+,d1					; load previous LAR >> 1
		asr.w	#$01,d1						; get >> 2
		add.w	d1,d0						; (pre >> 2) + (cur >> 2) + (cur >> 1)
		bsr.w	GSM_ReflectLAR					; perform reflection of LAR
		move.w	d0,(a3)+					; save to rrp
		dbf	d6,.Nexti					; repeat for all 8

		lea	(GSM_dp+27*2).w,a1				; k_start
		moveq	#(39-27),d6					; k_end
		bra.w	GSM_DoneLAR

		; -------------------------------------------------------
		; Interpolation of LARpp (4th coefficient set (40 to 159))
		;	for (int i = 1; i <= 8; i++)
		;	{
		;		short LAR = LARpp [cur] [i];
		;		rrp [i] = GSM_ReflectLAR (LAR);
		;	}
		;	k_start = 40;
		;	k_end = 159;
		; -------------------------------------------------------

GSM_LAR4:

	.Nexti:
		move.w	(a1)+,d0					; load current LAR
		bsr.w	GSM_ReflectLAR					; perform reflection of LAR
		move.w	d0,(a3)+					; save to rrp
		dbf	d6,.Nexti					; repeat for all 8

		lea	(GSM_dp+40*2).w,a1				; k_start
		moveq	#(159-40),d6					; k_end

GSM_DoneLAR:


	; ---------------------------------------------------------------------------
	; Single sub-frame loop (k_start to k_end samples per frame)
	; ---------------------------------------------------------------------------
	; This section deals with converting the dp samples into real samples
	; frame 1 =  0 to 12
	; frame 2 = 13 to 26
	; frame 3 = 27 to 39
	; frame 4 = 40 to 159
	; ---------------------------------------------------------------------------

SampleLoop:

		; ---------------------------------------------------------------------------
		; STS filtering, Deemphasis filtering, Upscale, Truncate, and Output
		; ---------------------------------------------------------------------------
		; This converts the "dp" sample results (i.e. reconstructed short term residual signal)
		; into the final 13-bit actual PCM sample, ready for playback
		; ---------------------------------------------------------------------------
		;	for (int k = k_start; k <= k_end; k++)
		;	{
		;		// Short term synthesis filtering section
		;		short Sample = dp [k];	 // always use the first 40 (first frame) for all 4 frames
		;		for (int i = 1; i <= 8; i++)
		;		{
		;			Sample -= (((int) rrp [9-i] * v [8-i]) + 0x4000) >> 0x0F;
		;			v [9-i] = v [8-i] + ((((int) rrp [9-i] * Sample) + 0x4000) >> 0x0F);
		;		}
		;		v [0] = Sample;
		;		// Deemphasis filtering
		;		Sample += (((int) msr * 28180) + 0x4000) >> 0x0F;
		;		msr = Sample;
		;		// Upscaling of the output signal
		;		Sample = GSM_add (Sample, Sample);	// This is the ONLY one which seems to need a cap (for obvious reasons).
		;		// Truncation of the output variable
		;		Sample &= 0xFFF8;
		;		// Output of the sample...
		;		Samples [SampPos++] = (float) Sample;
		;	}
		; ---------------------------------------------------------------------------

		lea	(GSM_v+8*2).w,a2				; start at v [8]
		lea	(GSM_rrp+8*2).w,a3				; start at rrp [9]

		move.w	(a1)+,d1					; load dp sample
		moveq	#8-1,d5						; rrp [1..8]

	.STS_Filter:
		move.w	-(a3),d2					; load rrp [9-i]
		move.w	-(a2),d3					; load v [8-i]
		move.w	d2,d0						; rrp [9-i] x v [8-i]
		muls.w	d3,d0						; ''
		add.l	d4,d0						; round up
		add.l	d0,d0						; get >> 1
		swap	d0						; get >> $F
		sub.w	d0,d1						; subtract from sample
		muls.w	d1,d2						; Sample x rrp [9-i]
		add.l	d4,d2						; round up
		add.l	d2,d2						; get >> 1
		swap	d2						; get >> $F
		add.w	d3,d2						; + v [8-i]
		move.w	d2,$02(a2)					; save to v [9-i]
		dbf	d5,.STS_Filter					; repeat for all rrp/v
		move.w	d1,(a2)						; save sample to v [0]

		; Deemphasis filtering

		move.w	(GSM_msr).w,d0					; msr x 28180
		muls.w	#28180,d0					; ''
		add.l	d4,d0						; round up
		add.l	d0,d0						; get >> 1
		swap	d0						; get >> $F

		add.w	d0,d1						; add to sample
		move.w	d1,(GSM_msr).w					; update msr for next sample

		asr.w	#$07,d1						; same as << 1, but right shift for 8-bit access only
		cmpi.w	#$007F,d1					; has it overflown?
		ble.s	.NoCapMax					; if not, continue
		moveq	#$7F,d1						; cap at maximum

	.NoCapMax:
		moveq	#$FFFFFF80,d2					; -$80 multi-use
		cmp.w	d2,d1						; has it underflown?
		bge.s	.NoCapMin					; if not, continue
		move.w	d2,d1						; cap at minimum

	.NoCapMin:
		add.b	d2,d1						; convert to unsigned

		; d1.b = sample

		; TODO:  Move the stop request up (perhaps above the muls 28180)
		; to avoid having to wait.
		move.w	#$0100,($A11100).l				; request Z80 stop (ON)
		btst.b	#$00,($A11100).l				; has the Z80 stopped yet?
		bne.s	*-$08						; if not, branch
		move.b	d1,(a5)+					; write sample
		move.w	#$0000,($A11100).l				; request Z80 stop (OFF)

		dbf	d6,SampleLoop					; repeat until k_end is reached

	; ---------------------------------------------------------------------------
	; loop back for next frame...
	; ---------------------------------------------------------------------------

		move.l	(sp)+,a2					; restore dp slot
		dbf	d7,FrameLoop					; repeat 4 times (one for each frame)

	; ---------------------------------------------------------------------------
	; End of GSM block
	; ---------------------------------------------------------------------------

		move.w	(GSM_LARppCur).w,d0				; swap LAR'' buffers
		move.w	(GSM_LARppPre).w,(GSM_LARppCur).w		; ''
		move.w	d0,(GSM_LARppPre).w				; ''

		move.l	a0,(GSM_Current).w				; update sample address
		rts							; finish block

; =============================================================================
; -----------------------------------------------------------------------------
; Computation of the rp[1..8] from the interpolated LARp[1..8]
; -----------------------------------------------------------------------------
; The input of this procedure is the interpolated LARp[1..8] array.
; The reflection coefficients, rp[i], are used in the analysis
; filter and in the synthesis filter.
; -----------------------------------------------------------------------------

GSM_ReflectLAR:
		bmi.s	.Neg						; if the coefficient was negative, branch for negative checks

	; --- Positive side ---

		cmpi.w	#11059,d0					; is it in approximation 1?
		bcs.s	.A1						; if so, branch
		cmpi.w	#20070,d0					; is it in approximation 2?
		bcs.s	.A2						; if so, branch
		asr.w	#$02,d0						; 26112/32767 = 0.796875 (1.625)
		addi.w	#26112,d0					; ''
		rts

	.A2:
		addi.w	#11059,d0					; 20070/32767 = 0.6125 = 1.225/2
		rts

	.A1:
		add.w	d0,d0						; 11059/32767 = 0.3375 = 0.675/2
		rts

	; --- Negative side ---

	.Neg:
		cmpi.w	#-11059+1,d0					; is it in approximation 1?
		bcc.s	.N1						; if so, branch
		cmpi.w	#-20070+1,d0					; is it in approximation 2?
		bcc.s	.N2						; if so, branch
		asr.w	#$02,d0						; 26112/32767 = 0.796875 (1.625)
		subi.w	#26112,d0					; ''
		rts

	.N2:
		subi.w	#11059,d0					; 20070/32767 = 0.6125 = 1.225/2
		rts

	.N1:
		add.w	d0,d0						; 11059/32767 = 0.3375 = 0.675/2
		rts

; ===========================================================================
; ---------------------------------------------------------------------------
; xMc load routines (one for each 3 bits)
; ---------------------------------------------------------------------------

		align	$100	; might need $1000

LoadxMc1:
		moveq	#%01110000,d1					; get xMc 1 bits
		and.b	(a0),d1						; ''
		lsr.b	#$03,d1						; align
		addi.w	#LoadxMc2-LoadxMc1,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMc2:
		moveq	#%00001110,d1					; get xMc 2 bits
		and.b	(a0),d1						; ''
		addi.w	#LoadxMc3-LoadxMc2,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMc3:
		moveq	#%00000001,d1					; get first xMc 3 bit
		and.b	(a0)+,d1					; ''
		move.b	(a0),d0						; get other two xMc bits
		add.b	d0,d0						; ''
		addx.b	d1,d1						; '' put with first xMc
		add.b	d0,d0						; ''
		addx.b	d1,d1						; '' put with second xMc
		add.b	d1,d1						; align
		addi.w	#LoadxMc4-LoadxMc3,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMc4:
		moveq	#%00111000,d1					; get xMc 4 bits
		and.b	(a0),d1						; ''
		lsr.b	#$02,d1						; align
		addi.w	#LoadxMc5-LoadxMc4,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMc5:
		moveq	#%00000111,d1					; get xMc 5 bits
		and.b	(a0)+,d1					; ''
		add.b	d1,d1						; align
		addi.w	#LoadxMc6-LoadxMc5,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMc6:
		move.b	(a0),d1						; get xMc 6 bits
		lsr.b	#$04,d1						; align
		andi.w	#%00001110,d1					; get only those bits (and clear upper byte)
		addi.w	#LoadxMc7-LoadxMc6,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMc7:
		moveq	#%00011100,d1					; get xMc 7 bits
		and.b	(a0),d1						; ''
		lsr.b	#$01,d1						; align
		addi.w	#LoadxMc8-LoadxMc7,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMc8:
		moveq	#%00000011,d1					; get first two xMc 8 bits
		and.b	(a0)+,d1					; ''
		move.b	(a0),d0						; get last xMc bit
		add.b	d0,d0						; ''
		addx.b	d1,d1						; '' put with first two xMc
		add.b	d1,d1						; align
		addi.w	#LoadxMc9-LoadxMc8,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMc9:
		moveq	#%01110000,d1					; get xMc 9 bits
		and.b	(a0),d1						; ''
		lsr.b	#$03,d1						; align
		addi.w	#LoadxMcA-LoadxMc9,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMcA:
		moveq	#%00001110,d1					; get xMc 10 bits
		and.b	(a0),d1						; ''
		addi.w	#LoadxMcB-LoadxMcA,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMcB:
		moveq	#%00000001,d1					; get first xMc 11 bit
		and.b	(a0)+,d1					; ''
		move.b	(a0),d0						; get other two xMc bits
		add.b	d0,d0						; ''
		addx.b	d1,d1						; '' put with first xMc
		add.b	d0,d0						; ''
		addx.b	d1,d1						; '' put with second xMc
		add.b	d1,d1						; align
		addi.w	#LoadxMcC-LoadxMcB,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMcC:
		moveq	#%00111000,d1					; get xMc 12 bits
		and.b	(a0),d1						; ''
		lsr.b	#$02,d1						; align
		addi.w	#LoadxMcD-LoadxMcC,(GSM_LoadxMc+2).w		; set next xMc routine
		rts							; ''

LoadxMcD:
		moveq	#%00000111,d1					; get xMc 13 bits
		and.b	(a0)+,d1					; ''
		add.b	d1,d1						; align
		moveq	#$FFFFFFFF,d3					; set no more xMc's
		rts							; ''

; ===========================================================================
; ---------------------------------------------------------------------------
; These are the results of LARpp(1..8) after calculating from LARc
; ---------------------------------------------------------------------------

		align	$100

LARpp1:
LARpp2:		dc.w	$999A,$9CCE,$A000,$A334,$A666,$A99A,$ACCE,$B000,$B334,$B666,$B99A,$BCCE,$C000,$C334,$C666,$C99A
		dc.w	$CCCE,$D000,$D334,$D666,$D99A,$DCCC,$E000,$E334,$E666,$E99A,$ECCC,$F000,$F334,$F666,$F99A,$FCCC
		dc.w	$0000,$0334,$0666,$099A,$0CCC,$1000,$1334,$1666,$199A,$1CCC,$2000,$2334,$2666,$299A,$2CCC,$3000
		dc.w	$3334,$3666,$399A,$3CCC,$4000,$4332,$4666,$499A,$4CCC,$5000,$5332,$5666,$599A,$5CCC,$6000,$6332

LARpp3:		dc.w	$C000,$C334,$C666,$C99A,$CCCE,$D000,$D334,$D666,$D99A,$DCCC,$E000,$E334,$E666,$E99A,$ECCC,$F000
		dc.w	$F334,$F666,$F99A,$FCCC,$0000,$0334,$0666,$099A,$0CCC,$1000,$1334,$1666,$199A,$1CCC,$2000,$2334

LARpp4:		dc.w	$DCCC,$E000,$E334,$E666,$E99A,$ECCC,$F000,$F334,$F666,$F99A,$FCCC,$0000,$0334,$0666,$099A,$0CCC
		dc.w	$1000,$1334,$1666,$199A,$1CCC,$2000,$2334,$2666,$299A,$2CCC,$3000,$3334,$3666,$399A,$3CCC,$4000

LARpp5:		dc.w	$D998,$DE4A,$E2FA,$E7AC,$EC5E,$F110,$F5C0,$FA72,$FF24,$03D4,$0886,$0D38,$11EA,$169A,$1B4C,$1FFE

LARpp6:		dc.w	$ECCC,$F112,$F556,$F99A,$FDDE,$0222,$0666,$0AAA,$0EEE,$1334,$1778,$1BBC,$2000,$2444,$2888,$2CCC

LARpp7:		dc.w	$E666,$EE14,$F5C2,$FD70,$051E,$0CCC,$147A,$1C26

LARpp8:		dc.w	$F332,$FA72,$01B4,$08F4,$1034,$1776,$1EB6,$25F6

	; --- bc to brp table ---

		; bc	00     01     10     11
QLB:		dc.w	$0CCD, $2CCD, $5333, $7FFF

; ---------------------------------------------------------------------------
; Look up tables for load LARc through arithmetic
; ---------------------------------------------------------------------------
ArithTable	macro	Shift, Mask, AddOff
AD = AddOff
MS = Mask
SH = Shift
	if SH>=0
Byte = 0
		rept	$100
		dc.b	((Byte>>SH)&MS)+AD
Byte = Byte+1
		endr
	else
SH = -SH
Byte = 0
		rept	$100
		dc.b	((Byte<<SH)&MS)+AD
Byte = Byte+1
		endr
	endif
		endm
; ---------------------------------------------------------------------------
		align	$100

LARcTables:	; This table is $200 bytes away from LARpp1

ASL3M01111000:	ArithTable	-3, %01111000, 0			; DDDD1111 to 01111000
ASR5M00000110:	ArithTable	 5, %00000110, LARpp1-LARpp1		; 11222222 to 00000110
ASL1M01111110:	ArithTable	-1, %01111110, LARpp2-LARpp1		; 11222222 to 02222220
ASR2M00111110:	ArithTable	 2, %00111110, LARpp3-LARpp1		; 33333444 to 00333330
ASL3M00111000:	ArithTable	-3, %00111000, 0			; 33333444 to 00444000
ASR5M000001102:	ArithTable	 5, %00000110, LARpp4-LARpp1		; 44555566 to 00000440
ASR1M00011110:	ArithTable	 1, %00011110, LARpp5-LARpp5		; 44555566 to 00055550
ASL3M00011000:	ArithTable	-3, %00011000, 0			; 44555566 to 00066000
ASR5M000001103:	ArithTable	 5, %00000110, LARpp6-LARpp5		; 66777888 to 00000660
ASR2M00001110:	ArithTable	 2, %00001110, LARpp7-LARpp5		; 66777888 to 00007770
ASL1M00001110:	ArithTable	-1, %00001110, LARpp8-LARpp5		; 66777888 to 00008880

; ---------------------------------------------------------------------------
; xMc/xmaxc to ep table value
; ---------------------------------------------------------------------------
;		xMc	  000   001   010   011   100   101   110   111
epval:		dc.w	$FFE4,$FFEC,$FFF4,$FFFC,$0004,$000C,$0014,$001C	; xmaxc = 000000
		dc.w	$FFC8,$FFD8,$FFE8,$FFF8,$0008,$0018,$0028,$0038	; xmaxc = 000001
		dc.w	$FFAC,$FFC4,$FFDC,$FFF4,$000C,$0024,$003C,$0054	; xmaxc = 000010
		dc.w	$FF90,$FFB0,$FFD0,$FFF0,$0010,$0030,$0050,$0070	; xmaxc = 000011
		dc.w	$FF74,$FF9C,$FFC4,$FFEC,$0014,$003C,$0064,$008C	; xmaxc = 000100
		dc.w	$FF58,$FF88,$FFB8,$FFE8,$0018,$0048,$0078,$00A8	; xmaxc = 000101
		dc.w	$FF3C,$FF74,$FFAC,$FFE4,$001C,$0054,$008C,$00C4	; xmaxc = 000110
		dc.w	$FF20,$FF60,$FFA0,$FFE0,$0020,$0060,$00A0,$00E0	; xmaxc = 000111
		dc.w	$FF04,$FF4C,$FF94,$FFDC,$0024,$006C,$00B4,$00FC	; xmaxc = 001000
		dc.w	$FEE8,$FF38,$FF88,$FFD8,$0028,$0078,$00C8,$0118	; xmaxc = 001001
		dc.w	$FECC,$FF24,$FF7C,$FFD4,$002C,$0084,$00DC,$0134	; xmaxc = 001010
		dc.w	$FEB0,$FF10,$FF70,$FFD0,$0030,$0090,$00F0,$0150	; xmaxc = 001011
		dc.w	$FE94,$FEFC,$FF64,$FFCC,$0034,$009C,$0104,$016C	; xmaxc = 001100
		dc.w	$FE78,$FEE8,$FF58,$FFC8,$0038,$00A8,$0118,$0188	; xmaxc = 001101
		dc.w	$FE5C,$FED4,$FF4C,$FFC4,$003C,$00B4,$012C,$01A4	; xmaxc = 001110
		dc.w	$FE40,$FEC0,$FF40,$FFC0,$0040,$00C0,$0140,$01C0	; xmaxc = 001111
		dc.w	$FE08,$FE98,$FF28,$FFB8,$0048,$00D8,$0168,$01F8	; xmaxc = 010000
		dc.w	$FDD0,$FE70,$FF10,$FFB0,$0050,$00F0,$0190,$0230	; xmaxc = 010001
		dc.w	$FD98,$FE48,$FEF8,$FFA8,$0058,$0108,$01B8,$0268	; xmaxc = 010010
		dc.w	$FD60,$FE20,$FEE0,$FFA0,$0060,$0120,$01E0,$02A0	; xmaxc = 010011
		dc.w	$FD28,$FDF8,$FEC8,$FF98,$0068,$0138,$0208,$02D8	; xmaxc = 010100
		dc.w	$FCF0,$FDD0,$FEB0,$FF90,$0070,$0150,$0230,$0310	; xmaxc = 010101
		dc.w	$FCB8,$FDA8,$FE98,$FF88,$0078,$0168,$0258,$0348	; xmaxc = 010110
		dc.w	$FC80,$FD80,$FE80,$FF80,$0080,$0180,$0280,$0380	; xmaxc = 010111
		dc.w	$FC10,$FD30,$FE50,$FF70,$0090,$01B0,$02D0,$03F0	; xmaxc = 011000
		dc.w	$FBA0,$FCE0,$FE20,$FF60,$00A0,$01E0,$0320,$0460	; xmaxc = 011001
		dc.w	$FB30,$FC90,$FDF0,$FF50,$00B0,$0210,$0370,$04D0	; xmaxc = 011010
		dc.w	$FAC0,$FC40,$FDC0,$FF40,$00C0,$0240,$03C0,$0540	; xmaxc = 011011
		dc.w	$FA50,$FBF0,$FD90,$FF30,$00D0,$0270,$0410,$05B0	; xmaxc = 011100
		dc.w	$F9E0,$FBA0,$FD60,$FF20,$00E0,$02A0,$0460,$0620	; xmaxc = 011101
		dc.w	$F970,$FB50,$FD30,$FF10,$00F0,$02D0,$04B0,$0690	; xmaxc = 011110
		dc.w	$F900,$FB00,$FD00,$FF00,$0100,$0300,$0500,$0700	; xmaxc = 011111
		dc.w	$F820,$FA60,$FCA0,$FEE0,$0120,$0360,$05A0,$07E0	; xmaxc = 100000
		dc.w	$F740,$F9C0,$FC40,$FEC0,$0140,$03C0,$0640,$08C0	; xmaxc = 100001
		dc.w	$F660,$F920,$FBE0,$FEA0,$0160,$0420,$06E0,$09A0	; xmaxc = 100010
		dc.w	$F580,$F880,$FB80,$FE80,$0180,$0480,$0780,$0A80	; xmaxc = 100011
		dc.w	$F4A0,$F7E0,$FB20,$FE60,$01A0,$04E0,$0820,$0B60	; xmaxc = 100100
		dc.w	$F3C0,$F740,$FAC0,$FE40,$01C0,$0540,$08C0,$0C40	; xmaxc = 100101
		dc.w	$F2E0,$F6A0,$FA60,$FE20,$01E0,$05A0,$0960,$0D20	; xmaxc = 100110
		dc.w	$F200,$F600,$FA00,$FE00,$0200,$0600,$0A00,$0E00	; xmaxc = 100111
		dc.w	$F040,$F4C0,$F940,$FDC0,$0240,$06C0,$0B40,$0FC0	; xmaxc = 101000
		dc.w	$EE80,$F380,$F880,$FD80,$0280,$0780,$0C80,$1180	; xmaxc = 101001
		dc.w	$ECC0,$F240,$F7C0,$FD40,$02C0,$0840,$0DC0,$1340	; xmaxc = 101010
		dc.w	$EB00,$F100,$F700,$FD00,$0300,$0900,$0F00,$1500	; xmaxc = 101011
		dc.w	$E940,$EFC0,$F640,$FCC0,$0340,$09C0,$1040,$16C0	; xmaxc = 101100
		dc.w	$E780,$EE80,$F580,$FC80,$0380,$0A80,$1180,$1880	; xmaxc = 101101
		dc.w	$E5C0,$ED40,$F4C0,$FC40,$03C0,$0B40,$12C0,$1A40	; xmaxc = 101110
		dc.w	$E400,$EC00,$F400,$FC00,$0400,$0C00,$1400,$1C00	; xmaxc = 101111
		dc.w	$E081,$E981,$F280,$FB80,$0480,$0D80,$1680,$1F80	; xmaxc = 110000
		dc.w	$DD01,$E701,$F100,$FB00,$0500,$0F00,$1900,$2300	; xmaxc = 110001
		dc.w	$D981,$E481,$EF80,$FA80,$0580,$1080,$1B80,$2680	; xmaxc = 110010
		dc.w	$D601,$E201,$EE00,$FA00,$0600,$1200,$1E00,$2A00	; xmaxc = 110011
		dc.w	$D281,$DF81,$EC80,$F980,$0680,$1380,$2080,$2D80	; xmaxc = 110100
		dc.w	$CF01,$DD01,$EB00,$F900,$0700,$1500,$2300,$3100	; xmaxc = 110101
		dc.w	$CB81,$DA81,$E980,$F880,$0780,$1680,$2580,$3480	; xmaxc = 110110
		dc.w	$C801,$D801,$E800,$F800,$0800,$1800,$2800,$3800	; xmaxc = 110111
		dc.w	$C101,$D301,$E500,$F700,$0900,$1B00,$2CFF,$3EFF	; xmaxc = 111000
		dc.w	$BA01,$CE01,$E200,$F600,$0A00,$1E00,$31FF,$45FF	; xmaxc = 111001
		dc.w	$B301,$C901,$DF00,$F500,$0B00,$2100,$36FF,$4CFF	; xmaxc = 111010
		dc.w	$AC01,$C401,$DC00,$F400,$0C00,$2400,$3BFF,$53FF	; xmaxc = 111011
		dc.w	$A501,$BF01,$D900,$F300,$0D00,$2700,$40FF,$5AFF	; xmaxc = 111100
		dc.w	$9E01,$BA01,$D600,$F200,$0E00,$2A00,$45FF,$61FF	; xmaxc = 111101
		dc.w	$9701,$B501,$D300,$F100,$0F00,$2D00,$4AFF,$68FF	; xmaxc = 111110
		dc.w	$9001,$B001,$D000,$F000,$1000,$3000,$4FFF,$6FFF	; xmaxc = 111111

; ===========================================================================
; ---------------------------------------------------------------------------
; Generic Z80 driver to playback small buffer supplied by 68k - MarkeyJester
; ---------------------------------------------------------------------------

Z80:		z80prog 0
		include	"Z80.asm"
		z80prog
Z80_End:

; ===========================================================================
; ---------------------------------------------------------------------------
; GSM Test sample
; ---------------------------------------------------------------------------

GSM_Sample:	incbin	"Sample.gsm"
		dc.b	$00
		even

; ===========================================================================

		align	$04					; must align to the next long-word (for checksum)
FinishRom:


















